// Service Worker for iPa Signer - Enhanced Offline Support
const CACHE_NAME = 'ipa-signer-v1.0.0';
const STATIC_CACHE = 'ipa-signer-static-v1.0.0';

// Files to cache for offline support
const STATIC_FILES = [
    '/',
    '/index.php',
    '/status.php',
    '/assets/css/style.css',
    '/assets/js/script_enhanced.js',
    '/assets/images/icon-512.svg',
    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css'
];

// Dynamic cache patterns
const CACHE_PATTERNS = {
    images: /\.(png|jpg|jpeg|gif|svg|webp)$/i,
    fonts: /\.(woff|woff2|ttf|eot)$/i,
    scripts: /\.(js)$/i,
    styles: /\.(css)$/i
};

// Install event - cache static files
self.addEventListener('install', (event) => {
    console.log('[SW] Installing...');
    
    event.waitUntil(
        Promise.all([
            caches.open(STATIC_CACHE).then((cache) => {
                console.log('[SW] Caching static files');
                return cache.addAll(STATIC_FILES.map(url => new Request(url, {
                    cache: 'reload'
                })));
            }),
            self.skipWaiting()
        ])
    );
});

// Activate event - clean up old caches
self.addEventListener('activate', (event) => {
    console.log('[SW] Activating...');
    
    event.waitUntil(
        Promise.all([
            // Clean up old caches
            caches.keys().then((cacheNames) => {
                return Promise.all(
                    cacheNames.map((cacheName) => {
                        if (cacheName !== CACHE_NAME && cacheName !== STATIC_CACHE) {
                            console.log('[SW] Deleting old cache:', cacheName);
                            return caches.delete(cacheName);
                        }
                    })
                );
            }),
            self.clients.claim()
        ])
    );
});

// Fetch event - serve from cache when offline
self.addEventListener('fetch', (event) => {
    const request = event.request;
    const url = new URL(request.url);
    
    // Skip non-GET requests
    if (request.method !== 'GET') {
        return;
    }
    
    // Skip external requests (except CDN)
    if (url.origin !== location.origin && !url.href.includes('cdnjs.cloudflare.com')) {
        return;
    }
    
    event.respondWith(
        handleRequest(request)
    );
});

async function handleRequest(request) {
    const url = new URL(request.url);
    
    try {
        // For main pages, try network first, fallback to cache
        if (isMainPage(url.pathname)) {
            return await networkFirstStrategy(request);
        }
        
        // For static assets, try cache first, fallback to network
        if (isStaticAsset(url.pathname)) {
            return await cacheFirstStrategy(request);
        }
        
        // For API endpoints, network only with offline fallback
        if (isApiEndpoint(url.pathname)) {
            return await networkOnlyWithFallback(request);
        }
        
        // Default: network first
        return await networkFirstStrategy(request);
        
    } catch (error) {
        console.error('[SW] Fetch error:', error);
        return await getOfflineFallback(request);
    }
}

// Network first strategy (for HTML pages)
async function networkFirstStrategy(request) {
    try {
        const networkResponse = await fetch(request);
        
        if (networkResponse.ok) {
            // Cache successful responses
            const cache = await caches.open(CACHE_NAME);
            cache.put(request, networkResponse.clone());
            return networkResponse;
        }
        
        throw new Error('Network response not ok');
    } catch (error) {
        console.log('[SW] Network failed, trying cache:', request.url);
        const cachedResponse = await caches.match(request);
        
        if (cachedResponse) {
            return cachedResponse;
        }
        
        throw error;
    }
}

// Cache first strategy (for static assets)
async function cacheFirstStrategy(request) {
    const cachedResponse = await caches.match(request);
    
    if (cachedResponse) {
        // Update cache in background
        updateCacheInBackground(request);
        return cachedResponse;
    }
    
    // Not in cache, fetch from network
    try {
        const networkResponse = await fetch(request);
        
        if (networkResponse.ok) {
            const cache = await caches.open(CACHE_NAME);
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.error('[SW] Failed to fetch asset:', request.url);
        throw error;
    }
}

// Network only with fallback (for API endpoints)
async function networkOnlyWithFallback(request) {
    try {
        const networkResponse = await fetch(request);
        return networkResponse;
    } catch (error) {
        console.log('[SW] API request failed, returning offline response');
        
        // Return offline response for API calls
        return new Response(
            JSON.stringify({
                success: false,
                message: 'Không có kết nối mạng. Vui lòng thử lại khi đã kết nối.',
                offline: true
            }),
            {
                status: 503,
                statusText: 'Service Unavailable',
                headers: {
                    'Content-Type': 'application/json'
                }
            }
        );
    }
}

// Update cache in background
function updateCacheInBackground(request) {
    fetch(request).then(response => {
        if (response.ok) {
            caches.open(CACHE_NAME).then(cache => {
                cache.put(request, response);
            });
        }
    }).catch(error => {
        console.log('[SW] Background update failed:', error);
    });
}

// Get offline fallback
async function getOfflineFallback(request) {
    const url = new URL(request.url);
    
    if (isMainPage(url.pathname)) {
        // Return cached index page as fallback
        const fallback = await caches.match('/index.php') || await caches.match('/');
        if (fallback) {
            return fallback;
        }
    }
    
    // Return generic offline page
    return new Response(
        `
        <!DOCTYPE html>
        <html>
        <head>
            <title>iPa Signer - Offline</title>
            <meta charset="utf-8">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <style>
                body {
                    font-family: -apple-system, BlinkMacSystemFont, sans-serif;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    min-height: 100vh;
                    margin: 0;
                    background: #f5f5f5;
                    color: #333;
                }
                .offline-container {
                    text-align: center;
                    padding: 2rem;
                    background: white;
                    border-radius: 12px;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
                    max-width: 400px;
                }
                .offline-icon {
                    font-size: 4rem;
                    margin-bottom: 1rem;
                }
                .offline-title {
                    font-size: 1.5rem;
                    margin-bottom: 1rem;
                    color: #666;
                }
                .offline-message {
                    color: #888;
                    line-height: 1.6;
                }
                .retry-button {
                    margin-top: 2rem;
                    padding: 0.75rem 2rem;
                    background: #007bff;
                    color: white;
                    border: none;
                    border-radius: 8px;
                    cursor: pointer;
                    font-size: 1rem;
                }
                .retry-button:hover {
                    background: #0056b3;
                }
            </style>
        </head>
        <body>
            <div class="offline-container">
                <div class="offline-icon">📱💫</div>
                <h1 class="offline-title">iPa Signer</h1>
                <p class="offline-message">
                    Bạn đang offline. Vui lòng kiểm tra kết nối mạng và thử lại.
                </p>
                <button class="retry-button" onclick="window.location.reload()">
                    Thử lại
                </button>
            </div>
        </body>
        </html>
        `,
        {
            status: 503,
            statusText: 'Service Unavailable',
            headers: {
                'Content-Type': 'text/html; charset=utf-8'
            }
        }
    );
}

// Helper functions
function isMainPage(pathname) {
    return pathname === '/' || 
           pathname === '/index.php' || 
           pathname.endsWith('.php') ||
           pathname.endsWith('.html');
}

function isStaticAsset(pathname) {
    return Object.values(CACHE_PATTERNS).some(pattern => pattern.test(pathname)) ||
           pathname.startsWith('/assets/');
}

function isApiEndpoint(pathname) {
    return pathname.includes('process.php') ||
           pathname.includes('status.php') ||
           pathname.includes('get_token.php') ||
           pathname.includes('manifest.php');
}

// Handle messages from main thread
self.addEventListener('message', (event) => {
    if (event.data && event.data.type === 'SKIP_WAITING') {
        self.skipWaiting();
    }
    
    if (event.data && event.data.type === 'GET_VERSION') {
        event.ports[0].postMessage({
            version: CACHE_NAME
        });
    }
    
    if (event.data && event.data.type === 'CLEAR_CACHE') {
        caches.keys().then(cacheNames => {
            return Promise.all(
                cacheNames.map(cacheName => caches.delete(cacheName))
            );
        }).then(() => {
            event.ports[0].postMessage({
                success: true
            });
        });
    }
});

// Background sync for queued requests (if supported)
self.addEventListener('sync', (event) => {
    if (event.tag === 'background-sync') {
        event.waitUntil(doBackgroundSync());
    }
});

async function doBackgroundSync() {
    // Implement background sync logic for queued signing requests
    console.log('[SW] Background sync triggered');
    
    // This could be extended to handle queued IPA signing requests
    // when the user comes back online
}

console.log('[SW] Service Worker loaded successfully');
