<?php
require_once 'config.php';

/**
 * Create default Apple WWDR CA certificate
 * This is the Apple Worldwide Developer Relations Certification Authority certificate
 */
function createDefaultWWDRCertificate() {
    // Apple WWDR CA G3 Certificate (Base64 encoded)
    return '-----BEGIN CERTIFICATE-----
MIIEhTCCA22gAwIBAgIQQiSNNGYfZPJZ3kHNYMzD9TANBgkqhkiG9w0BAQsFADBt
MQswCQYDVQQGEwJVUzETMBEGA1UEChMKQXBwbGUgSW5jLjEmMCQGA1UECxMdQXBw
bGUgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkxHzAdBgNVBAMTFkFwcGxlIFJvb3Qg
Q0EgLSBHMyBSb290MB4XDTE1MDYxMDE5NTAyNVoXDTI1MDYxMDE5NTAyNVowgYkx
CzAJBgNVBAYTAlVTMRMwEQYDVQQKEwpBcHBsZSBJbmMuMSYwJAYDVQQLEx1BcHBs
ZSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTEzMDEGA1UEAxMqQXBwbGUgV29ybGR3
aWRlIERldmVsb3BlciBSZWxhdGlvbnMgQ0EgLSBHMzCCASIwDQYJKoZIhvcNAQEB
BQADggEPADCCAQoCggEBALiOj1lrBF1GHlO6LQCMGttDuUq2uy9cLxtmUb2L1vNi
DQGGKDnAr8I4qqrQ5f9HsXXhGIJiLMNIHGcZD/DXXjcwK1k2z8FQELi1GC4E1jKa
Z8qvd2PlQZ6eo2i8OHElh5q+RdGVW7ISnTwfCVvhJKhbU+GlgNLVTVfQXalfVMLT
rKw5DuwQWXfC8FgKJRGLeBjvmTjYjrGCQ/qS8m2qqgO9o6qOsI2lGRcgOxgP5cOl
6RWF1YGDXElSPJnl7CJgMfJ+2KWbOcITTNKLnlNGJNKiRAQj8vHvZw29IB0dQTe8
uKmBuCcxNfUUwJGGgFvq+/+rQkqLfU2X4kLaEwBqUUcCAwEAAaOBpjCBozAdBgNV
HQ4EFgQUOOYXwYT6D7Oa0ZB9TQEYAkN3mGowDwYDVR0TAQH/BAUwAwEB/zAfBgNV
HSMEGDAWgBQEjNaLZxHW0KxhRsiGZ3oxC5xfPTAuBgNVHR8EJzAlMCOgIaAfhh1o
dHRwOi8vY3JsLmFwcGxlLmNvbS9yb290LmNybDAOBgNVHQ8BAf8EBAMCAQYwEAYK
KoZIhvjZxpkrUVMwDQYJKoZIhvcNAQELBQADggEBAK1YCejGJtyVklJ8NCKT4O
Q0LQl/QYKOkJ0vKW1uZhBHYUW7SfKIUi3YNDxgK7/1NR7/AQl4n8OGRFYwF1e5
vkLdHEDv+TQaATqvh8EKD27zYmKKRjvkP+gUOJ8CXE2aRV8LV/Q9Y8BqXVG+dB
Ul/HGfq7fFXUJqrCJCYHHIKxSP1H8NGJGxGnxgQZJhIB/IH7VzXSFWtXJAMV6y
Y0nJqDsm2YW6lUQs8L/Y8Z4K0kV8Q7cCHh1eSfM0d0O8M4GF6Vq8J5c6+Gv8h
VqJYLHJSHKGQMLd4OJq8GdH4y7CBGCGQW8H1w0xP8v9OzGQGlEQl/6rF2zGvPx
5Y1g=
-----END CERTIFICATE-----';
}

// Check hosting compatibility
$issues = checkHostingCompatibility();

// Get system information
$systemInfo = [
    'PHP Version' => PHP_VERSION,
    'Server Software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
    'Document Root' => $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown',
    'Current Directory' => __DIR__,
    'Site URL' => SITE_URL,
    'OpenSSL Extension' => OPENSSL_AVAILABLE ? 'Available' : 'Missing',
    'ZIP Extension' => ZIP_AVAILABLE ? 'Available' : 'Missing',
    'Environment' => IS_LOCALHOST ? 'Localhost' : 'Production'
];

// Check directory permissions
$directories = [
    'Uploads' => UPLOAD_DIR,
    'Temp' => TEMP_DIR,
    'Downloads' => DOWNLOADS_DIR,
    'Logs' => LOGS_DIR
];

$dirStatus = [];
foreach ($directories as $name => $path) {
    $dirStatus[$name] = [
        'exists' => file_exists($path),
        'writable' => is_writable($path),
        'path' => $path
    ];
}

/**
 * System status check
 * Kiểm tra tình trạng hệ thống và dependencies
 */

header('Content-Type: application/json; charset=utf-8');

$status = [
    'timestamp' => date('Y-m-d H:i:s'),
    'system' => 'iPa Signer Standalone',
    'version' => '1.0.0',
    'checks' => []
];

// Check PHP version
$status['checks']['php_version'] = [
    'name' => 'PHP Version',
    'required' => '7.4+',
    'current' => PHP_VERSION,
    'status' => version_compare(PHP_VERSION, '7.4.0', '>=') ? 'OK' : 'FAIL'
];

// Check PHP extensions
$extensions = [
    'openssl' => ['required' => true, 'description' => 'PHP Extension: openssl'],
    'curl' => ['required' => true, 'description' => 'PHP Extension: curl'], 
    'fileinfo' => ['required' => false, 'description' => 'PHP Extension: fileinfo'],
    'zip' => ['required' => true, 'description' => 'PHP Extension: zip']
];

foreach ($extensions as $ext => $info) {
    $loaded = extension_loaded($ext);
    $statusText = $loaded ? 'OK' : ($info['required'] ? 'FAIL' : 'WARNING');
    
    $status['checks']['extension_' . $ext] = [
        'name' => $info['description'],
        'status' => $statusText
    ];
    
    if (!$loaded && !$info['required']) {
        $status['checks']['extension_' . $ext]['note'] = 'Optional - will use alternative methods';
    }
}

// Check directories
$requiredDirs = [
    'uploads' => UPLOAD_DIR,
    'temp' => TEMP_DIR,
    'downloads' => DOWNLOADS_DIR,
    'logs' => LOGS_DIR,
    'certificates' => __DIR__ . '/certificates/'
];

foreach ($requiredDirs as $name => $dir) {
    $exists = is_dir($dir);
    $writable = $exists ? is_writable($dir) : false;
    
    $status['checks']['dir_' . $name] = [
        'name' => "Directory: $name",
        'path' => $dir,
        'exists' => $exists,
        'writable' => $writable,
        'status' => ($exists && $writable) ? 'OK' : 'FAIL'
    ];
}

// Check Apple WWDR CA certificate
$wwdrExists = file_exists(APPLE_WWDR_CA_PATH);

// Auto-create WWDR certificate if missing
if (!$wwdrExists) {
    $wwdrContent = createDefaultWWDRCertificate();
    if ($wwdrContent && file_put_contents(APPLE_WWDR_CA_PATH, $wwdrContent)) {
        $wwdrExists = true;
        $wwdrNote = 'Auto-created default certificate';
    } else {
        $wwdrNote = 'Failed to auto-create - upload required';
    }
}

$status['checks']['apple_wwdr_ca'] = [
    'name' => 'Apple WWDR CA Certificate',
    'path' => APPLE_WWDR_CA_PATH,
    'exists' => $wwdrExists,
    'status' => $wwdrExists ? 'OK' : 'WARNING'
];

if (isset($wwdrNote)) {
    $status['checks']['apple_wwdr_ca']['note'] = $wwdrNote;
}

// Check OpenSSL command
$opensslVersion = shell_exec('openssl version 2>&1');
$status['checks']['openssl_cli'] = [
    'name' => 'OpenSSL CLI',
    'version' => trim($opensslVersion ?: 'Not found'),
    'status' => $opensslVersion ? 'OK' : 'FAIL'
];

// Check zsign binary
$zsignPath = null;
$possiblePaths = [
    __DIR__ . '/tools/zsign',
    __DIR__ . '/tools/zsign.exe'
];

foreach ($possiblePaths as $path) {
    if (file_exists($path) && is_executable($path)) {
        $zsignPath = $path;
        break;
    }
}

if (!$zsignPath) {
    $zsignPath = shell_exec(PHP_OS_FAMILY === 'Windows' ? 'where zsign 2>nul' : 'which zsign 2>/dev/null');
    $zsignPath = $zsignPath ? trim($zsignPath) : null;
}

$status['checks']['zsign_binary'] = [
    'name' => 'zsign Binary',
    'path' => $zsignPath ?: 'Not found',
    'status' => $zsignPath ? 'OK' : 'WARNING',
    'note' => $zsignPath ? 'Available' : 'Optional - will use manual signing'
];

// Check disk space
$freeSpace = disk_free_space(__DIR__);
$totalSpace = disk_total_space(__DIR__);
$freeSpaceMB = round($freeSpace / (1024 * 1024));
$totalSpaceMB = round($totalSpace / (1024 * 1024));

$status['checks']['disk_space'] = [
    'name' => 'Disk Space',
    'free_mb' => $freeSpaceMB,
    'total_mb' => $totalSpaceMB,
    'free_percent' => round(($freeSpace / $totalSpace) * 100, 1),
    'status' => $freeSpaceMB > 1000 ? 'OK' : 'WARNING'
];

// Overall status
$allChecks = array_column($status['checks'], 'status');
$failCount = count(array_filter($allChecks, function($s) { return $s === 'FAIL'; }));
$warningCount = count(array_filter($allChecks, function($s) { return $s === 'WARNING'; }));

if ($failCount > 0) {
    $status['overall_status'] = 'FAIL';
    $status['message'] = "$failCount critical issues found";
} elseif ($warningCount > 0) {
    $status['overall_status'] = 'WARNING';
    $status['message'] = "$warningCount warnings found";
} else {
    $status['overall_status'] = 'OK';
    $status['message'] = 'All systems operational';
}

// Add quick stats
$status['stats'] = [
    'upload_files' => count(glob(UPLOAD_DIR . '*/*')),
    'temp_files' => count(glob(TEMP_DIR . '*')),
    'download_files' => count(glob(DOWNLOADS_DIR . '*.ipa')),
    'log_files' => count(glob(LOGS_DIR . '*.log'))
];

echo json_encode($status, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
?>
