<?php
require_once 'config.php';
require_once 'classes/FileManager.php';
require_once 'classes/OCSPValidator.php';
require_once 'classes/IPASigner.php';

// Enhanced security and logging
session_start();

// Rate limiting check
if (!checkRateLimit()) {
    logAccess('rate_limit_exceeded');
    http_response_code(429);
    echo json_encode([
        'success' => false,
        'message' => 'Quá nhiều request. Vui lòng thử lại sau 1 phút.'
    ]);
    exit;
}

// Log access
logAccess('ipa_signing_attempt');

// Enhanced error reporting
if (IS_PRODUCTION) {
    error_reporting(0);
    ini_set('display_errors', 0);
} else {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
}

// Set JSON response header
header('Content-Type: application/json; charset=utf-8');

// Enhanced security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');

try {
    // Check if request method is POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Chỉ chấp nhận POST request');
    }

    // Security token validation (if implemented)
    if (isset($_POST['security_token'])) {
        if (!validateSecurityToken($_POST['security_token'])) {
            throw new Exception('Token bảo mật không hợp lệ');
        }
    }

    // Enhanced file validation
    if (!isset($_FILES['ipa_file']) || !isset($_FILES['p12_file']) || !isset($_FILES['mobileprovision_file'])) {
        throw new Exception('Thiếu file upload. Vui lòng chọn đầy đủ IPA, P12 và Mobileprovision');
    }

    // Validate P12 password
    if (!isset($_POST['p12_password']) || empty(trim($_POST['p12_password']))) {
        throw new Exception('Vui lòng nhập mật khẩu cho chứng chỉ P12');
    }

    $password = trim($_POST['p12_password']);

    // Log signing attempt
    logAccess('file_upload_start', json_encode([
        'ipa_size' => $_FILES['ipa_file']['size'] ?? 0,
        'p12_size' => $_FILES['p12_file']['size'] ?? 0,
        'mp_size' => $_FILES['mobileprovision_file']['size'] ?? 0
    ]));

    // Initialize FileManager with enhanced error handling
    $fileManager = new FileManager();

    // Handle file uploads with detailed logging
    try {
        $ipaPath = $fileManager->handleUpload($_FILES['ipa_file'], 'ipa');
        logAccess('ipa_upload_success', basename($ipaPath));
    } catch (Exception $e) {
        logAccess('ipa_upload_failed', $e->getMessage());
        throw new Exception('Lỗi upload file IPA: ' . $e->getMessage());
    }

    try {
        $p12Path = $fileManager->handleUpload($_FILES['p12_file'], 'p12');
        logAccess('p12_upload_success', basename($p12Path));
    } catch (Exception $e) {
        logAccess('p12_upload_failed', $e->getMessage());
        throw new Exception('Lỗi upload file P12: ' . $e->getMessage());
    }

    try {
        $mpPath = $fileManager->handleUpload($_FILES['mobileprovision_file'], 'mobileprovision');
        logAccess('mp_upload_success', basename($mpPath));
    } catch (Exception $e) {
        logAccess('mp_upload_failed', $e->getMessage());
        throw new Exception('Lỗi upload file Mobileprovision: ' . $e->getMessage());
    }

    // Enhanced OCSP validation with logging
    try {
        $ocspValidator = new OCSPValidator();
        logAccess('ocsp_validation_start');
        
        $ocspResult = $ocspValidator->validateCertificate($p12Path, $password);
        
        if (!$ocspResult['success']) {
            logAccess('ocsp_validation_failed', $ocspResult['message']);
            throw new Exception('Xác thực OCSP thất bại: ' . $ocspResult['message']);
        }

        if ($ocspResult['is_revoked']) {
            logAccess('certificate_revoked');
            throw new Exception('Chứng chỉ đã bị thu hồi. Không thể sử dụng để ký.');
        }
        
        logAccess('ocsp_validation_success');
    } catch (Exception $e) {
        logAccess('ocsp_error', $e->getMessage());
        throw new Exception('Lỗi xác thực chứng chỉ: ' . $e->getMessage());
    }

    // Enhanced IPA signing with detailed logging
    try {
        $signer = new IPASigner();
        logAccess('signing_start');
        
        $signResult = $signer->signIPA($ipaPath, $p12Path, $mpPath, $password);

        if (!$signResult['success']) {
            logAccess('signing_failed', $signResult['message']);
            throw new Exception('Ký IPA thất bại: ' . $signResult['message']);
        }
        
        // Log successful signing
        $fileSize = file_exists($signResult['signed_file']) ? filesize($signResult['signed_file']) : 0;
        logAccess('signing_success', json_encode([
            'signed_file' => basename($signResult['signed_file']),
            'file_size' => $fileSize
        ]));
        
    } catch (Exception $e) {
        logAccess('signing_error', $e->getMessage());
        throw new Exception('Lỗi trong quá trình ký: ' . $e->getMessage());
    }

    // Enhanced success response with metadata
    $timestamp = date('Y-m-d_H-i-s');
    $fileSize = file_exists($signResult['signed_file']) ? filesize($signResult['signed_file']) : 0;
    
    $response = [
        'success' => true,
        'message' => 'Ký IPA thành công',
        'signed_file' => basename($signResult['signed_file']),
        'download_url' => 'downloads/' . basename($signResult['signed_file']),
        'install_url' => 'itms-services://?action=download-manifest&url=' . urlencode(SITE_URL . '/manifest.php?file=' . basename($signResult['signed_file'])),
        'file_size' => formatFileSize($fileSize),
        'signed_at' => $timestamp,
        'cert_info' => $ocspResult['cert_info'] ?? null
    ];
    
    echo json_encode($response);
    
    // Log successful completion
    logAccess('process_completed', json_encode([
        'signed_file' => basename($signResult['signed_file']),
        'file_size' => $fileSize
    ]));

} catch (Exception $e) {
    // Enhanced error handling and logging
    $errorMessage = $e->getMessage();
    
    // Log detailed error
    logAccess('process_error', $errorMessage);
    
    // In production, sanitize sensitive error messages
    if (IS_PRODUCTION) {
        if (strpos($errorMessage, 'upload') !== false) {
            $errorMessage = 'Lỗi upload file. Vui lòng kiểm tra file và thử lại.';
        } elseif (strpos($errorMessage, 'OCSP') !== false || strpos($errorMessage, 'chứng chỉ') !== false) {
            $errorMessage = 'Lỗi xác thực chứng chỉ. Vui lòng kiểm tra file P12 và mật khẩu.';
        } elseif (strpos($errorMessage, 'ký') !== false) {
            $errorMessage = 'Lỗi trong quá trình ký. Vui lòng thử lại.';
        } else {
            $errorMessage = 'Có lỗi xảy ra. Vui lòng thử lại sau.';
        }
    }
    
    // Log error for debugging
    error_log('IPA Signing Error: ' . $e->getMessage());
    
    // Return enhanced error response
    $errorResponse = [
        'success' => false,
        'message' => $errorMessage,
        'timestamp' => date('Y-m-d H:i:s'),
        'error_code' => 'SIGNING_ERROR'
    ];
    
    echo json_encode($errorResponse);
}

// Helper function for file size formatting
function formatFileSize($size) {
    if ($size <= 0) return '0 B';
    $units = ['B', 'KB', 'MB', 'GB'];
    $power = floor(log($size) / log(1024));
    return round($size / pow(1024, $power), 2) . ' ' . $units[$power];
}
?>
