<?php
require_once 'config.php';

// Enhanced manifest generation with security and logging
header('Content-Type: application/xml; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Security headers for iOS manifest
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

// Log manifest request
logAccess('manifest_request', $_GET['file'] ?? 'unknown');

// Enhanced input validation
if (!isset($_GET['file']) || empty($_GET['file'])) {
    http_response_code(400);
    logAccess('manifest_error', 'Missing file parameter');
    exit('Invalid request: missing file parameter');
}

$fileName = basename($_GET['file']);

// Validate file exists and is safe
$downloadPath = DOWNLOADS_DIR . '/' . $fileName;
if (!file_exists($downloadPath)) {
    http_response_code(404);
    logAccess('manifest_error', 'File not found: ' . $fileName);
    exit('File not found');
}

// Validate file extension
if (!preg_match('/\.ipa$/i', $fileName)) {
    http_response_code(400);
    logAccess('manifest_error', 'Invalid file type: ' . $fileName);
    exit('Invalid file type');
}

// Generate secure URLs
$downloadUrl = SITE_URL . '/downloads/' . $fileName;
$iconUrl = SITE_URL . '/assets/images/icon-512.svg';

// Enhanced file information
$fileSize = filesize($downloadPath);
$lastModified = filemtime($downloadPath);

// Log successful manifest generation
logAccess('manifest_generated', json_encode([
    'file' => $fileName,
    'size' => $fileSize,
    'modified' => date('Y-m-d H:i:s', $lastModified)
]));

// Extract app info from IPA (basic implementation)
$appInfo = extractAppInfo($downloadPath);

echo '<?xml version="1.0" encoding="UTF-8"?>';
?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>items</key>
    <array>
        <dict>
            <key>assets</key>
            <array>
                <dict>
                    <key>kind</key>
                    <string>software-package</string>
                    <key>url</key>
                    <string><?php echo htmlspecialchars($downloadUrl); ?></string>
                </dict>
                <?php if (!empty($iconUrl)): ?>
                <dict>
                    <key>kind</key>
                    <string>display-image</string>
                    <key>url</key>
                    <string><?php echo htmlspecialchars($iconUrl); ?></string>
                </dict>
                <dict>
                    <key>kind</key>
                    <string>full-size-image</string>
                    <key>url</key>
                    <string><?php echo htmlspecialchars($iconUrl); ?></string>
                </dict>
                <?php endif; ?>
            </array>
            <key>metadata</key>
            <dict>
                <key>bundle-identifier</key>
                <string><?php echo htmlspecialchars($appInfo['bundle_id'] ?? 'com.ipasigner.app'); ?></string>
                <key>bundle-version</key>
                <string><?php echo htmlspecialchars($appInfo['version'] ?? '1.0'); ?></string>
                <key>kind</key>
                <string>software</string>
                <key>title</key>
                <string><?php echo htmlspecialchars($appInfo['name'] ?? 'Signed App'); ?></string>
                <key>subtitle</key>
                <string>Signed with iPa Signer</string>
                <key>platform-identifier</key>
                <string>com.apple.platform.iphoneos</string>
                <key>minimum-os-version</key>
                <string><?php echo htmlspecialchars($appInfo['min_os'] ?? '10.0'); ?></string>
            </dict>
        </dict>
    </array>
</dict>
</plist>
<?php

/**
 * Extract basic app information from IPA file
 * This is a simplified implementation - could be enhanced with proper plist parsing
 */
function extractAppInfo($ipaPath) {
    $appInfo = [
        'name' => 'Signed App',
        'bundle_id' => 'com.ipasigner.app',
        'version' => '1.0',
        'min_os' => '10.0'
    ];
    
    try {
        // Basic implementation - could be enhanced to actually read Info.plist from IPA
        $zip = new ZipArchive();
        if ($zip->open($ipaPath) === TRUE) {
            // Look for Info.plist in the app bundle
            for ($i = 0; $i < $zip->numFiles; $i++) {
                $filename = $zip->getNameIndex($i);
                if (preg_match('/Payload\/[^\/]+\.app\/Info\.plist$/', $filename)) {
                    $plistContent = $zip->getFromIndex($i);
                    if ($plistContent) {
                        // Parse basic info from plist
                        $appInfo = parsePlistInfo($plistContent);
                    }
                    break;
                }
            }
            $zip->close();
        }
    } catch (Exception $e) {
        // Fallback to default values
        logAccess('app_info_extraction_failed', $e->getMessage());
    }
    
    return $appInfo;
}

/**
 * Parse basic information from Info.plist content
 * Simplified XML parsing - could use proper plist parser
 */
function parsePlistInfo($plistContent) {
    $appInfo = [
        'name' => 'Signed App',
        'bundle_id' => 'com.ipasigner.app',
        'version' => '1.0',
        'min_os' => '10.0'
    ];
    
    try {
        // Simple XML parsing for basic info
        if (preg_match('/<key>CFBundleDisplayName<\/key>\s*<string>([^<]+)<\/string>/', $plistContent, $matches)) {
            $appInfo['name'] = $matches[1];
        } elseif (preg_match('/<key>CFBundleName<\/key>\s*<string>([^<]+)<\/string>/', $plistContent, $matches)) {
            $appInfo['name'] = $matches[1];
        }
        
        if (preg_match('/<key>CFBundleIdentifier<\/key>\s*<string>([^<]+)<\/string>/', $plistContent, $matches)) {
            $appInfo['bundle_id'] = $matches[1];
        }
        
        if (preg_match('/<key>CFBundleShortVersionString<\/key>\s*<string>([^<]+)<\/string>/', $plistContent, $matches)) {
            $appInfo['version'] = $matches[1];
        }
        
        if (preg_match('/<key>MinimumOSVersion<\/key>\s*<string>([^<]+)<\/string>/', $plistContent, $matches)) {
            $appInfo['min_os'] = $matches[1];
        }
    } catch (Exception $e) {
        logAccess('plist_parsing_failed', $e->getMessage());
    }
    
    return $appInfo;
}
?>
?>
