<?php
/**
 * Configuration for cPanel/Shared Hosting
 * Auto-detects environment and sets appropriate paths and URLs
 */

// Auto-detect hosting environment
$isLocalhost = in_array($_SERVER['HTTP_HOST'], ['localhost', '127.0.0.1']) || 
               strpos($_SERVER['HTTP_HOST'], 'localhost:') === 0;

// Site URL configuration
if ($isLocalhost) {
    // Local development
    define('SITE_URL', 'http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['SCRIPT_NAME']));
} else {
    // Production hosting - auto-detect domain
    $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $path = dirname($_SERVER['SCRIPT_NAME']);
    define('SITE_URL', $protocol . '://' . $host . $path);
}

// Directory paths - works for both localhost and cPanel
define('BASE_DIR', __DIR__);
define('UPLOAD_DIR', BASE_DIR . '/uploads/');
define('TEMP_DIR', BASE_DIR . '/temp/');
define('DOWNLOADS_DIR', BASE_DIR . '/downloads/');
define('LOGS_DIR', BASE_DIR . '/logs/');

// File size limits (conservative for shared hosting)
define('MAX_IPA_SIZE', 50 * 1024 * 1024);  // 50MB (reduced for shared hosting)
define('MAX_P12_SIZE', 5 * 1024 * 1024);   // 5MB
define('MAX_MP_SIZE', 512 * 1024);         // 512KB

// Cleanup settings
define('CLEANUP_INTERVAL_MINUTES', 30);
define('CLEANUP_LOG_FILE', LOGS_DIR . 'cleanup.log');

// Apple WWDR CA certificate path
define('APPLE_WWDR_CA_PATH', BASE_DIR . '/certificates/AppleWWDRCAG3.pem');

// PHP settings for shared hosting compatibility
ini_set('max_execution_time', 300);    // 5 minutes
ini_set('memory_limit', '256M');       // 256MB
ini_set('upload_max_filesize', '50M'); // 50MB
ini_set('post_max_size', '55M');       // 55MB

// Create necessary directories with proper permissions
$dirs = [UPLOAD_DIR, TEMP_DIR, DOWNLOADS_DIR, LOGS_DIR];
foreach ($dirs as $dir) {
    if (!file_exists($dir)) {
        if (!mkdir($dir, 0755, true)) {
            error_log("Failed to create directory: $dir");
        }
    }
    
    // Ensure directory is writable
    if (!is_writable($dir)) {
        chmod($dir, 0755);
    }
}

// Create subdirectories for uploads
$uploadSubdirs = ['ipa', 'p12', 'mobileprovision'];
foreach ($uploadSubdirs as $subdir) {
    $path = UPLOAD_DIR . $subdir;
    if (!file_exists($path)) {
        if (!mkdir($path, 0755, true)) {
            error_log("Failed to create upload subdirectory: $path");
        }
    }
}

// Environment info (for debugging)
define('IS_LOCALHOST', $isLocalhost);
define('PHP_VERSION_COMPAT', version_compare(PHP_VERSION, '7.4.0', '>='));
define('OPENSSL_AVAILABLE', extension_loaded('openssl'));
define('ZIP_AVAILABLE', extension_loaded('zip'));

// Error reporting for production
if (!$isLocalhost) {
    // Production environment - log errors, don't display
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', LOGS_DIR . 'error.log');
} else {
    // Development environment - show errors
    ini_set('display_errors', 1);
    ini_set('log_errors', 1);
    error_reporting(E_ALL);
}

// Function to check hosting compatibility
function checkHostingCompatibility() {
    $issues = [];
    
    if (!PHP_VERSION_COMPAT) {
        $issues[] = 'PHP version ' . PHP_VERSION . ' may not be fully compatible. Recommended: 7.4+';
    }
    
    if (!OPENSSL_AVAILABLE) {
        $issues[] = 'OpenSSL extension is not available';
    }
    
    if (!ZIP_AVAILABLE) {
        $issues[] = 'ZIP extension is not available';
    }
    
    foreach ([UPLOAD_DIR, TEMP_DIR, DOWNLOADS_DIR, LOGS_DIR] as $dir) {
        if (!is_writable($dir)) {
            $issues[] = "Directory not writable: $dir";
        }
    }
    
    return $issues;
}
?>
