<?php
/**
 * Advanced Configuration for cPanel/Shared Hosting
 * Auto-detects environment and sets appropriate paths and URLs
 */

// Advanced environment detection
function detectEnvironment() {
    $indicators = [
        'localhost' => ['localhost', '127.0.0.1', '::1'],
        'development' => ['dev.', 'test.', 'staging.'],
        'cpanel' => ['cpanel', 'whm', 'shared'],
    ];
    
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $software = strtolower($_SERVER['SERVER_SOFTWARE'] ?? '');
    
    // Check for localhost
    foreach ($indicators['localhost'] as $local) {
        if (strpos($host, $local) !== false) {
            return 'localhost';
        }
    }
    
    // Check for development environment
    foreach ($indicators['development'] as $dev) {
        if (strpos($host, $dev) !== false) {
            return 'development';
        }
    }
    
    // Check for cPanel indicators
    if (strpos($software, 'apache') !== false || 
        strpos($software, 'nginx') !== false ||
        isset($_SERVER['CPANEL_VERSION'])) {
        return 'production';
    }
    
    return 'production'; // Default to production
}

// Smart URL detection with subdirectory support
function detectSiteUrl() {
    $protocol = 'http';
    
    // Enhanced HTTPS detection
    if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
        $protocol = 'https';
    } elseif (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') {
        $protocol = 'https';
    } elseif (isset($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on') {
        $protocol = 'https';
    } elseif (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443) {
        $protocol = 'https';
    }
    
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    // Smart path detection
    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    
    // Remove script filename from path
    $path = dirname($scriptName);
    
    // Normalize path
    $path = str_replace('\\', '/', $path);
    $path = rtrim($path, '/');
    
    // Handle subdirectory installations
    if ($path === '.' || $path === '') {
        $path = '';
    }
    
    return $protocol . '://' . $host . $path;
}

// Environment configuration
$environment = detectEnvironment();
define('ENVIRONMENT', $environment);
define('IS_LOCALHOST', $environment === 'localhost');
define('IS_PRODUCTION', $environment === 'production');

// Site URL with smart detection
define('SITE_URL', detectSiteUrl());

// Base directory detection
define('BASE_DIR', __DIR__);

// Directory paths - works for all environments
define('UPLOAD_DIR', BASE_DIR . '/uploads/');
define('TEMP_DIR', BASE_DIR . '/temp/');
define('DOWNLOADS_DIR', BASE_DIR . '/downloads/');
define('LOGS_DIR', BASE_DIR . '/logs/');
define('CERTIFICATES_DIR', BASE_DIR . '/certificates/');

// Environment-specific file size limits
if (IS_LOCALHOST) {
    // Development - larger limits
    define('MAX_IPA_SIZE', 100 * 1024 * 1024);  // 100MB
    define('MAX_P12_SIZE', 10 * 1024 * 1024);   // 10MB
    define('MAX_MP_SIZE', 1024 * 1024);         // 1MB
} else {
    // Production - conservative limits for shared hosting
    define('MAX_IPA_SIZE', 50 * 1024 * 1024);   // 50MB
    define('MAX_P12_SIZE', 5 * 1024 * 1024);    // 5MB
    define('MAX_MP_SIZE', 512 * 1024);          // 512KB
}

// Enhanced cleanup settings
define('CLEANUP_INTERVAL_MINUTES', IS_LOCALHOST ? 60 : 30); // Longer for dev
define('CLEANUP_LOG_FILE', LOGS_DIR . 'cleanup.log');
define('ERROR_LOG_FILE', LOGS_DIR . 'error.log');
define('ACCESS_LOG_FILE', LOGS_DIR . 'access.log');

// Security settings
define('MAX_LOGIN_ATTEMPTS', 5);
define('SECURITY_TOKEN_EXPIRY', 3600); // 1 hour

// Apple WWDR CA certificate path
define('APPLE_WWDR_CA_PATH', CERTIFICATES_DIR . 'AppleWWDRCAG3.pem');

// Advanced PHP settings for shared hosting compatibility
function configurePhpSettings() {
    if (!IS_LOCALHOST) {
        // Production settings
        @ini_set('max_execution_time', 300);
        @ini_set('memory_limit', '256M');
        @ini_set('upload_max_filesize', '50M');
        @ini_set('post_max_size', '55M');
        @ini_set('max_file_uploads', 20);
        
        // Error handling for production
        @ini_set('display_errors', 0);
        @ini_set('log_errors', 1);
        @ini_set('error_log', ERROR_LOG_FILE);
    } else {
        // Development settings
        @ini_set('display_errors', 1);
        @ini_set('log_errors', 1);
        error_reporting(E_ALL);
        @ini_set('max_execution_time', 0); // No limit for dev
    }
    
    // Common settings
    @ini_set('session.cookie_httponly', 1);
    @ini_set('session.cookie_secure', IS_PRODUCTION ? 1 : 0);
    @ini_set('session.use_strict_mode', 1);
}

// Enhanced directory creation with proper permissions
function createDirectoriesWithPermissions() {
    $dirs = [
        UPLOAD_DIR => 0755,
        TEMP_DIR => 0755,
        DOWNLOADS_DIR => 0755,
        LOGS_DIR => 0755,
        CERTIFICATES_DIR => 0644
    ];
    
    foreach ($dirs as $dir => $perm) {
        if (!file_exists($dir)) {
            if (!@mkdir($dir, $perm, true)) {
                error_log("Failed to create directory: $dir");
                continue;
            }
        }
        
        // Ensure proper permissions
        if (!@chmod($dir, $perm)) {
            error_log("Failed to set permissions for: $dir");
        }
        
        // Add security .htaccess for upload directories
        if (in_array($dir, [UPLOAD_DIR, TEMP_DIR, DOWNLOADS_DIR])) {
            $htaccessContent = "Options -Indexes\n";
            $htaccessContent .= "Order Deny,Allow\n";
            $htaccessContent .= "Deny from all\n";
            $htaccessContent .= "<Files \"*.ipa\">\n";
            $htaccessContent .= "    Allow from all\n";
            $htaccessContent .= "</Files>\n";
            
            $htaccessFile = $dir . '.htaccess';
            if (!file_exists($htaccessFile)) {
                @file_put_contents($htaccessFile, $htaccessContent);
            }
        }
    }
    
    // Create upload subdirectories
    $uploadSubdirs = ['ipa', 'p12', 'mobileprovision'];
    foreach ($uploadSubdirs as $subdir) {
        $path = UPLOAD_DIR . $subdir;
        if (!file_exists($path)) {
            @mkdir($path, 0755, true);
        }
    }
}

// Security token system
function generateSecurityToken() {
    if (!isset($_SESSION)) {
        session_start();
    }
    
    $token = bin2hex(random_bytes(32));
    $_SESSION['security_token'] = $token;
    $_SESSION['token_time'] = time();
    
    return $token;
}

function validateSecurityToken($token) {
    if (!isset($_SESSION)) {
        session_start();
    }
    
    if (!isset($_SESSION['security_token']) || 
        !isset($_SESSION['token_time']) ||
        (time() - $_SESSION['token_time']) > SECURITY_TOKEN_EXPIRY) {
        return false;
    }
    
    return hash_equals($_SESSION['security_token'], $token);
}

// Rate limiting
function checkRateLimit($identifier = null) {
    if (!$identifier) {
        $identifier = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
    
    $rateLimitFile = TEMP_DIR . 'rate_limit_' . md5($identifier) . '.tmp';
    $currentTime = time();
    $maxRequests = IS_LOCALHOST ? 100 : 10; // Requests per minute
    $timeWindow = 60; // 1 minute
    
    if (file_exists($rateLimitFile)) {
        $data = json_decode(file_get_contents($rateLimitFile), true);
        
        // Clean old entries
        $data = array_filter($data, function($timestamp) use ($currentTime, $timeWindow) {
            return ($currentTime - $timestamp) < $timeWindow;
        });
        
        if (count($data) >= $maxRequests) {
            return false; // Rate limit exceeded
        }
    } else {
        $data = [];
    }
    
    $data[] = $currentTime;
    file_put_contents($rateLimitFile, json_encode($data));
    
    return true;
}

// Initialize configuration
configurePhpSettings();
createDirectoriesWithPermissions();

// Environment info (for debugging)
define('PHP_VERSION_COMPAT', version_compare(PHP_VERSION, '7.4.0', '>='));
define('OPENSSL_AVAILABLE', extension_loaded('openssl'));
define('ZIP_AVAILABLE', extension_loaded('zip'));
define('CURL_AVAILABLE', extension_loaded('curl'));

// Enhanced compatibility check
function checkHostingCompatibility() {
    $issues = [];
    $warnings = [];
    
    if (!PHP_VERSION_COMPAT) {
        $issues[] = 'PHP version ' . PHP_VERSION . ' may not be fully compatible. Recommended: 7.4+';
    }
    
    if (!OPENSSL_AVAILABLE) {
        $issues[] = 'OpenSSL extension is not available - P12 certificate processing will fail';
    }
    
    if (!ZIP_AVAILABLE) {
        $issues[] = 'ZIP extension is not available - IPA file processing will fail';
    }
    
    if (!CURL_AVAILABLE) {
        $warnings[] = 'CURL extension not available - some features may be limited';
    }
    
    // Check directory permissions
    foreach ([UPLOAD_DIR, TEMP_DIR, DOWNLOADS_DIR, LOGS_DIR] as $dir) {
        if (!is_writable($dir)) {
            $issues[] = "Directory not writable: $dir";
        }
    }
    
    // Check Apple WWDR certificate
    if (!file_exists(APPLE_WWDR_CA_PATH)) {
        $issues[] = "Apple WWDR certificate not found: " . APPLE_WWDR_CA_PATH;
    }
    
    // Check upload limits
    $uploadLimit = ini_get('upload_max_filesize');
    $postLimit = ini_get('post_max_size');
    
    if (parseSize($uploadLimit) < MAX_IPA_SIZE) {
        $warnings[] = "Upload limit ($uploadLimit) may be too small for IPA files";
    }
    
    if (parseSize($postLimit) < MAX_IPA_SIZE) {
        $warnings[] = "POST limit ($postLimit) may be too small for IPA files";
    }
    
    return ['issues' => $issues, 'warnings' => $warnings];
}

function parseSize($size) {
    $unit = preg_replace('/[^bkmgtpezy]/i', '', $size);
    $size = preg_replace('/[^0-9\.]/', '', $size);
    if ($unit) {
        return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
    }
    return round($size);
}

// Access logging for security
function logAccess($action = 'page_view', $details = '') {
    if (!IS_LOCALHOST) {
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'action' => $action,
            'details' => $details,
            'url' => $_SERVER['REQUEST_URI'] ?? '',
            'referer' => $_SERVER['HTTP_REFERER'] ?? ''
        ];
        
        $logLine = json_encode($logEntry) . "\n";
        @file_put_contents(ACCESS_LOG_FILE, $logLine, FILE_APPEND | LOCK_EX);
    }
}

// Auto-cleanup system enhancement
function shouldRunCleanup() {
    $lastCleanupFile = LOGS_DIR . 'last_cleanup.txt';
    $lastCleanup = file_exists($lastCleanupFile) ? (int)file_get_contents($lastCleanupFile) : 0;
    $now = time();
    
    return ($now - $lastCleanup) > (CLEANUP_INTERVAL_MINUTES * 60);
}

function runAutomaticCleanup() {
    if (!shouldRunCleanup()) {
        return false;
    }
    
    $cleanupDirs = [
        UPLOAD_DIR => ['*.ipa', '*.p12', '*.mobileprovision'],
        TEMP_DIR => ['*'],
        DOWNLOADS_DIR => ['*.ipa']
    ];
    
    $cutoffTime = time() - (CLEANUP_INTERVAL_MINUTES * 60);
    $deletedFiles = 0;
    $deletedSize = 0;
    
    foreach ($cleanupDirs as $dir => $patterns) {
        if (!is_dir($dir)) continue;
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile() && $file->getMTime() < $cutoffTime) {
                $size = $file->getSize();
                if (@unlink($file->getPathname())) {
                    $deletedFiles++;
                    $deletedSize += $size;
                }
            }
        }
    }
    
    // Log cleanup results
    $logMessage = sprintf(
        "[%s] Auto-cleanup: %d files deleted, %.2f MB freed\n",
        date('Y-m-d H:i:s'),
        $deletedFiles,
        $deletedSize / (1024 * 1024)
    );
    
    @file_put_contents(CLEANUP_LOG_FILE, $logMessage, FILE_APPEND | LOCK_EX);
    @file_put_contents(LOGS_DIR . 'last_cleanup.txt', time());
    
    return true;
}

// Initialize cleanup on every request (lightweight check)
if (rand(1, 100) <= 10) { // 10% chance to run cleanup check
    runAutomaticCleanup();
}
?>
foreach ($uploadSubdirs as $subdir) {
    $path = UPLOAD_DIR . $subdir . '/';
    if (!file_exists($path)) {
        mkdir($path, 0755, true);
    }
}

// Error handling
function logError($message) {
    $logFile = LOGS_DIR . 'error.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message" . PHP_EOL, FILE_APPEND | LOCK_EX);
}

// Set error handler
set_error_handler(function($severity, $message, $file, $line) {
    logError("PHP Error: $message in $file on line $line");
});

// Set exception handler  
set_exception_handler(function($exception) {
    logError("Uncaught Exception: " . $exception->getMessage() . " in " . $exception->getFile() . " on line " . $exception->getLine());
});
?>
