<?php
require_once 'config.php';

/**
 * Cleanup scheduled task
 * This script should be run periodically (e.g., via cron job)
 */

function cleanupFiles() {
    $cleanupDirs = [
        UPLOAD_DIR,
        TEMP_DIR,
        DOWNLOADS_DIR
    ];
    
    $cutoffTime = time() - (CLEANUP_INTERVAL_MINUTES * 60);
    $totalDeleted = 0;
    $totalSize = 0;
    
    foreach ($cleanupDirs as $dir) {
        $result = cleanupDirectory($dir, $cutoffTime);
        $totalDeleted += $result['files'];
        $totalSize += $result['size'];
    }
    
    // Log cleanup results
    $logMessage = sprintf(
        "[%s] Scheduled cleanup: %d files deleted, %.2f MB freed\n",
        date('Y-m-d H:i:s'),
        $totalDeleted,
        $totalSize / (1024 * 1024)
    );
    
    file_put_contents(CLEANUP_LOG_FILE, $logMessage, FILE_APPEND | LOCK_EX);
    
    return [
        'files_deleted' => $totalDeleted,
        'size_freed' => $totalSize
    ];
}

function cleanupDirectory($dir, $cutoffTime) {
    $deletedFiles = 0;
    $deletedSize = 0;
    
    if (!is_dir($dir)) {
        return ['files' => 0, 'size' => 0];
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    
    foreach ($iterator as $file) {
        if ($file->isFile() && $file->getMTime() < $cutoffTime) {
            $size = $file->getSize();
            $filePath = $file->getPathname();
            
            // Skip log files and config files
            if (strpos($filePath, '/logs/') !== false || 
                strpos($filePath, 'config.php') !== false ||
                strpos($filePath, '.htaccess') !== false) {
                continue;
            }
            
            if (unlink($filePath)) {
                $deletedFiles++;
                $deletedSize += $size;
            }
        }
    }
    
    return ['files' => $deletedFiles, 'size' => $deletedSize];
}

// Run cleanup if called directly
if (basename(__FILE__) == basename($_SERVER['SCRIPT_NAME'])) {
    $result = cleanupFiles();
    
    if (php_sapi_name() === 'cli') {
        // Command line output
        echo "Cleanup completed:\n";
        echo "Files deleted: " . $result['files_deleted'] . "\n";
        echo "Size freed: " . round($result['size_freed'] / (1024 * 1024), 2) . " MB\n";
    } else {
        // Web output
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'files_deleted' => $result['files_deleted'],
            'size_freed_mb' => round($result['size_freed'] / (1024 * 1024), 2)
        ]);
    }
}
?>
