<?php
require_once __DIR__ . '/../config.php';

class IPASigner {
    
    private $zsignPath;
    private $toolsDir;
    
    public function __construct() {
        $this->toolsDir = __DIR__ . '/../tools/';
        // Path to zsign binary (you'll need to compile/download zsign for your system)
        $this->zsignPath = $this->findZsignBinary();
    }
    
    /**
     * Sign IPA file
     */
    public function signIPA($ipaPath, $p12Path, $mobileprovisionPath, $password) {
        try {
            // Validate inputs
            $this->validateInputs($ipaPath, $p12Path, $mobileprovisionPath);
            
            // Extract certificate and private key from P12
            $certData = $this->extractP12Contents($p12Path, $password);
            
            // Validate provisioning profile compatibility
            $this->validateProvisioningProfile($mobileprovisionPath, $certData['cert_content']);
            
            // Generate output filename
            $outputPath = $this->generateOutputPath($ipaPath);
            
            // Perform signing
            $result = $this->performSigning($ipaPath, $certData['cert_path'], $certData['key_path'], $mobileprovisionPath, $outputPath, $password);
            
            // Cleanup temporary files
            $this->cleanup([$certData['cert_path'], $certData['key_path']]);
            
            if ($result['success']) {
                return [
                    'success' => true,
                    'signed_file' => $outputPath,
                    'message' => 'IPA signed successfully'
                ];
            } else {
                throw new Exception($result['message']);
            }
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Find zsign binary with improved paths
     */
    private function findZsignBinary() {
        $possiblePaths = [
            $this->toolsDir . 'zsign',
            $this->toolsDir . 'zsign.exe',
            __DIR__ . '/../tools/zsign',
            __DIR__ . '/../tools/zsign.exe',
            '/usr/local/bin/zsign',
            '/usr/bin/zsign'
        ];
        
        foreach ($possiblePaths as $path) {
            if (file_exists($path) && is_executable($path)) {
                return $path;
            }
        }
        
        // Try to find in PATH
        $command = PHP_OS_FAMILY === 'Windows' ? 'where zsign' : 'which zsign';
        $output = shell_exec($command);
        
        if ($output && trim($output)) {
            return trim($output);
        }
        
        // zsign not found, will use manual signing
        return null;
    }
    
    /**
     * Validate input files
     */
    private function validateInputs($ipaPath, $p12Path, $mobileprovisionPath) {
        if (!file_exists($ipaPath)) {
            throw new Exception('IPA file not found');
        }
        
        if (!file_exists($p12Path)) {
            throw new Exception('P12 certificate file not found');
        }
        
        if (!file_exists($mobileprovisionPath)) {
            throw new Exception('Mobileprovision file not found');
        }
        
        // Validate file sizes
        if (filesize($ipaPath) > MAX_IPA_SIZE) {
            throw new Exception('IPA file too large');
        }
    }
    
    /**
     * Extract certificate and private key from P12 file using PHP OpenSSL
     */
    private function extractP12Contents($p12Path, $password) {
        $certPath = TEMP_DIR . 'cert_' . uniqid() . '.pem';
        $keyPath = TEMP_DIR . 'key_' . uniqid() . '.pem';
        
        try {
            // Read P12 file content
            $p12Content = file_get_contents($p12Path);
            if (!$p12Content) {
                throw new Exception('Không thể đọc file P12');
            }
            
            // Parse PKCS12 using PHP OpenSSL
            $certs = [];
            if (!openssl_pkcs12_read($p12Content, $certs, $password)) {
                // Try with empty password if provided password fails
                if (!openssl_pkcs12_read($p12Content, $certs, '')) {
                    throw new Exception('Mật khẩu P12 không đúng hoặc file P12 bị hỏng');
                }
            }
            
            if (!isset($certs['cert']) || empty($certs['cert'])) {
                throw new Exception('Không tìm thấy certificate trong file P12');
            }
            
            if (!isset($certs['pkey']) || empty($certs['pkey'])) {
                throw new Exception('Không tìm thấy private key trong file P12');
            }
            
            // Save certificate to file
            if (!file_put_contents($certPath, $certs['cert'])) {
                throw new Exception('Không thể lưu certificate tạm thời');
            }
            
            // Save private key to file
            if (!file_put_contents($keyPath, $certs['pkey'])) {
                unlink($certPath);
                throw new Exception('Không thể lưu private key tạm thời');
            }
            
            return [
                'cert_path' => $certPath,
                'key_path' => $keyPath,
                'cert_content' => $certs['cert']
            ];
            
        } catch (Exception $e) {
            // Cleanup on error
            if (file_exists($certPath)) unlink($certPath);
            if (file_exists($keyPath)) unlink($keyPath);
            
            error_log("P12 extraction failed: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Validate provisioning profile compatibility
     */
    private function validateProvisioningProfile($mobileprovisionPath, $certContent) {
        $mpContent = file_get_contents($mobileprovisionPath);
        
        if (!$mpContent) {
            throw new Exception('Cannot read mobileprovision file');
        }
        
        // Extract embedded plist from mobileprovision
        $startPos = strpos($mpContent, '<?xml');
        $endPos = strrpos($mpContent, '</plist>');
        
        if ($startPos === false || $endPos === false) {
            throw new Exception('Invalid mobileprovision format');
        }
        
        $plistContent = substr($mpContent, $startPos, $endPos - $startPos + 8);
        
        // Save to temporary file for parsing
        $tempPlistPath = TEMP_DIR . 'mp_' . uniqid() . '.plist';
        file_put_contents($tempPlistPath, $plistContent);
        
        try {
            // Parse plist (this is a simplified version - you might want to use a proper plist parser)
            $plistData = $this->parsePlist($tempPlistPath);
            
            // Check if certificate is in the provisioning profile
            if (!$this->isCertificateInProfile($certContent, $plistData)) {
                throw new Exception('Certificate does not match provisioning profile');
            }
            
            // Check if provisioning profile has not expired
            if (isset($plistData['ExpirationDate'])) {
                $expirationDate = strtotime($plistData['ExpirationDate']);
                if (time() > $expirationDate) {
                    throw new Exception('Provisioning profile has expired');
                }
            }
            
        } finally {
            if (file_exists($tempPlistPath)) {
                unlink($tempPlistPath);
            }
        }
    }
    
    /**
     * Simple plist parser (basic implementation)
     */
    private function parsePlist($plistPath) {
        // This is a basic implementation. For production, consider using a proper plist parser
        $content = file_get_contents($plistPath);
        
        // Extract ExpirationDate
        if (preg_match('/<key>ExpirationDate<\/key>\s*<date>([^<]+)<\/date>/', $content, $matches)) {
            $data['ExpirationDate'] = $matches[1];
        }
        
        // Extract DeveloperCertificates (this is simplified)
        if (preg_match('/<key>DeveloperCertificates<\/key>\s*<array>(.*?)<\/array>/s', $content, $matches)) {
            $data['DeveloperCertificates'] = $matches[1];
        }
        
        return $data ?? [];
    }
    
    /**
     * Check if certificate is in provisioning profile
     */
    private function isCertificateInProfile($certContent, $plistData) {
        // This is a simplified check. In production, you should properly compare certificate thumbprints
        if (!isset($plistData['DeveloperCertificates'])) {
            return false;
        }
        
        // Extract certificate info for comparison
        $cert = openssl_x509_parse($certContent);
        if (!$cert) {
            return false;
        }
        
        // For now, we'll assume it's valid if we can parse both
        // In production, you should compare certificate fingerprints
        return true;
    }
    
    /**
     * Generate output path for signed IPA
     */
    private function generateOutputPath($ipaPath) {
        $basename = pathinfo($ipaPath, PATHINFO_FILENAME);
        $timestamp = date('Y-m-d_H-i-s');
        $outputPath = DOWNLOADS_DIR . $basename . '_signed_' . $timestamp . '.ipa';
        
        // Ensure downloads directory exists and is writable
        if (!file_exists(DOWNLOADS_DIR)) {
            if (!mkdir(DOWNLOADS_DIR, 0755, true)) {
                throw new Exception('Không thể tạo thư mục downloads');
            }
        }
        
        if (!is_writable(DOWNLOADS_DIR)) {
            throw new Exception('Thư mục downloads không có quyền ghi');
        }
        
        return $outputPath;
    }
    
    /**
     * Perform the actual signing
     */
    private function performSigning($ipaPath, $certPath, $keyPath, $mobileprovisionPath, $outputPath, $password) {
        // If zsign is available, use it
        if ($this->zsignPath) {
            return $this->signWithZsign($ipaPath, $certPath, $keyPath, $mobileprovisionPath, $outputPath);
        } else {
            // Simple fallback - copy file for testing
            return $this->simpleFallback($ipaPath, $outputPath);
        }
    }
    
    /**
     * Simple fallback method for testing when zsign is not available
     */
    private function simpleFallback($ipaPath, $outputPath) {
        try {
            if (!copy($ipaPath, $outputPath)) {
                throw new Exception('Không thể copy file IPA');
            }
            
            if (!file_exists($outputPath)) {
                throw new Exception('File đầu ra không được tạo');
            }
            
            return [
                'success' => true,
                'message' => 'File processed successfully (testing mode - no actual signing)'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Sign using zsign tool
     */
    private function signWithZsign($ipaPath, $certPath, $keyPath, $mobileprovisionPath, $outputPath) {
        $command = sprintf(
            '"%s" -c "%s" -k "%s" -m "%s" -o "%s" "%s" 2>&1',
            $this->zsignPath,
            $certPath,
            $keyPath,
            $mobileprovisionPath,
            $outputPath,
            $ipaPath
        );
        
        $output = [];
        $returnCode = 0;
        exec($command, $output, $returnCode);
        
        if ($returnCode === 0 && file_exists($outputPath)) {
            return [
                'success' => true,
                'message' => 'Signed successfully with zsign'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'zsign failed: ' . implode(' ', $output)
            ];
        }
    }
    
    /**
     * Manual signing process (simplified version)
     */
    private function signManually($ipaPath, $certPath, $keyPath, $mobileprovisionPath, $outputPath) {
        try {
            // Extract IPA
            $extractDir = TEMP_DIR . 'ipa_extract_' . uniqid();
            mkdir($extractDir, 0755, true);
            
            $extractCommand = sprintf(
                'cd "%s" && unzip -q "%s"',
                $extractDir,
                $ipaPath
            );
            
            exec($extractCommand, $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception('Failed to extract IPA');
            }
            
            // Find app bundle
            $payloadDir = $extractDir . '/Payload';
            if (!is_dir($payloadDir)) {
                throw new Exception('Invalid IPA structure - no Payload directory');
            }
            
            $appDirs = glob($payloadDir . '/*.app');
            if (empty($appDirs)) {
                throw new Exception('Invalid IPA structure - no .app bundle found');
            }
            
            $appDir = $appDirs[0];
            
            // Copy provisioning profile
            copy($mobileprovisionPath, $appDir . '/embedded.mobileprovision');
            
            // Create entitlements file (simplified)
            $entitlementsPath = TEMP_DIR . 'entitlements_' . uniqid() . '.plist';
            $this->createEntitlements($entitlementsPath, $mobileprovisionPath);
            
            // Sign the app bundle
            $signCommand = sprintf(
                'codesign --force --sign "%s" --entitlements "%s" --timestamp=none "%s" 2>&1',
                $this->getCertificateIdentity($certPath),
                $entitlementsPath,
                $appDir
            );
            
            exec($signCommand, $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception('Code signing failed: ' . implode(' ', $output));
            }
            
            // Repackage IPA using PHP ZipArchive
            if (!$this->createZipFromDirectory($extractDir, $outputPath)) {
                throw new Exception('Không thể lưu file ipa đã ký');
            }
            
            if (!file_exists($outputPath)) {
                throw new Exception('File IPA đầu ra không được tạo thành công');
            }
            
            // Cleanup
            $this->recursiveDelete($extractDir);
            if (file_exists($entitlementsPath)) {
                unlink($entitlementsPath);
            }
            
            return [
                'success' => true,
                'message' => 'Signed successfully with manual process'
            ];
            
        } catch (Exception $e) {
            // Cleanup on error
            if (isset($extractDir) && is_dir($extractDir)) {
                $this->recursiveDelete($extractDir);
            }
            if (isset($entitlementsPath) && file_exists($entitlementsPath)) {
                unlink($entitlementsPath);
            }
            
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get certificate identity for codesign
     */
    private function getCertificateIdentity($certPath) {
        $certContent = file_get_contents($certPath);
        $cert = openssl_x509_parse($certContent);
        
        if ($cert && isset($cert['subject']['CN'])) {
            return $cert['subject']['CN'];
        }
        
        return 'iPhone Developer'; // Fallback
    }
    
    /**
     * Create entitlements file
     */
    private function createEntitlements($entitlementsPath, $mobileprovisionPath) {
        // This is a simplified entitlements creation
        // In production, you should extract entitlements from the provisioning profile
        $entitlements = '<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>application-identifier</key>
    <string>*</string>
    <key>get-task-allow</key>
    <true/>
</dict>
</plist>';
        
        file_put_contents($entitlementsPath, $entitlements);
    }
    
    /**
     * Recursive directory deletion
     */
    private function recursiveDelete($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . DIRECTORY_SEPARATOR . $file;
            is_dir($path) ? $this->recursiveDelete($path) : unlink($path);
        }
        rmdir($dir);
    }
    
    /**
     * Create ZIP archive from directory using PHP ZipArchive
     */
    private function createZipFromDirectory($sourceDir, $outputPath) {
        if (!extension_loaded('zip')) {
            throw new Exception('PHP Zip extension is required');
        }
        
        $zip = new ZipArchive();
        $result = $zip->open($outputPath, ZipArchive::CREATE | ZipArchive::OVERWRITE);
        
        if ($result !== TRUE) {
            return false;
        }
        
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($sourceDir),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = substr($filePath, strlen($sourceDir) + 1);
                
                // Convert Windows path separators to forward slashes for ZIP
                $relativePath = str_replace('\\', '/', $relativePath);
                
                $zip->addFile($filePath, $relativePath);
            }
        }
        
        $result = $zip->close();
        return $result;
    }
    
    /**
     * Cleanup temporary files
     */
    private function cleanup($files) {
        foreach ($files as $file) {
            if (file_exists($file)) {
                unlink($file);
            }
        }
    }
}
?>
