<?php
require_once __DIR__ . '/../config.php';

class FileManager {
    
    public function __construct() {
        $this->scheduleCleanup();
    }
    
    /**
     * Handle file upload
     */
    public function handleUpload($file, $type) {
        if (!isset($file) || $file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception("Lỗi upload file $type: " . $this->getUploadErrorMessage($file['error']));
        }
        
        // Validate file type and size
        $this->validateFile($file, $type);
        
        // Generate unique filename
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = uniqid($type . '_') . '.' . $extension;
        $uploadPath = UPLOAD_DIR . $type . '/' . $filename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
            throw new Exception("Không thể lưu file $type");
        }
        
        // Set proper permissions
        chmod($uploadPath, 0644);
        
        return $uploadPath;
    }
    
    /**
     * Validate uploaded file
     */
    private function validateFile($file, $type) {
        $allowedTypes = [
            'ipa' => ['application/octet-stream', 'application/zip'],
            'p12' => ['application/x-pkcs12', 'application/octet-stream'],
            'mobileprovision' => ['application/octet-stream', 'text/plain']
        ];
        
        $allowedExtensions = [
            'ipa' => ['ipa'],
            'p12' => ['p12'],
            'mobileprovision' => ['mobileprovision']
        ];
        
        $maxSizes = [
            'ipa' => MAX_IPA_SIZE,
            'p12' => MAX_P12_SIZE,
            'mobileprovision' => MAX_MP_SIZE
        ];
        
        // Check file size
        if ($file['size'] > $maxSizes[$type]) {
            $maxSizeMB = round($maxSizes[$type] / (1024 * 1024));
            throw new Exception("File $type quá lớn. Kích thước tối đa: {$maxSizeMB}MB");
        }
        
        // Check file extension
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($extension, $allowedExtensions[$type])) {
            throw new Exception("File $type không đúng định dạng. Chỉ chấp nhận: " . implode(', ', $allowedExtensions[$type]));
        }
        
        // Additional security checks
        $this->performSecurityChecks($file, $type);
    }
    
    /**
     * Perform additional security checks
     */
    private function performSecurityChecks($file, $type) {
        // Check for executable files
        $dangerousExtensions = ['php', 'php3', 'php4', 'php5', 'phtml', 'exe', 'bat', 'cmd', 'com', 'scr', 'vbs', 'js', 'jar'];
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (in_array($extension, $dangerousExtensions)) {
            throw new Exception("File không được phép: $extension");
        }
        
        // Check file content for basic validation
        if ($type === 'ipa') {
            $this->validateIPAFile($file['tmp_name']);
        } elseif ($type === 'p12') {
            $this->validateP12File($file['tmp_name']);
        } elseif ($type === 'mobileprovision') {
            $this->validateMobileprovisionFile($file['tmp_name']);
        }
    }
    
    /**
     * Validate IPA file
     */
    private function validateIPAFile($filePath) {
        // Read first few bytes to check ZIP signature
        $handle = fopen($filePath, 'rb');
        if ($handle) {
            $header = fread($handle, 4);
            fclose($handle);
            
            // ZIP file signature
            if ($header !== "PK\x03\x04") {
                throw new Exception("File IPA không hợp lệ (không phải ZIP archive)");
            }
        }
    }
    
    /**
     * Validate P12 file
     */
    private function validateP12File($filePath) {
        // Check file size
        if (filesize($filePath) < 50) {
            throw new Exception("File P12 quá nhỏ (có thể bị hỏng)");
        }
        
        // Simple validation - just check if file exists and has reasonable size
        // The real validation will happen when we try to extract the certificate
        $content = file_get_contents($filePath, false, null, 0, 10);
        
        // Check if it's completely empty or text file
        if (empty(trim($content)) || ctype_print($content)) {
            throw new Exception("File P12 không hợp lệ. Vui lòng chọn file certificate đúng định dạng .p12");
        }
    }
    
    /**
     * Validate mobileprovision file
     */
    private function validateMobileprovisionFile($filePath) {
        $content = file_get_contents($filePath, false, null, 0, 1024);
        if (strpos($content, '<?xml') === false || strpos($content, 'plist') === false) {
            throw new Exception("File mobileprovision không hợp lệ (không phải plist format)");
        }
    }
    
    /**
     * Get upload error message
     */
    private function getUploadErrorMessage($errorCode) {
        $messages = [
            UPLOAD_ERR_INI_SIZE => 'File quá lớn (server limit)',
            UPLOAD_ERR_FORM_SIZE => 'File quá lớn (form limit)',
            UPLOAD_ERR_PARTIAL => 'File upload không hoàn tất',
            UPLOAD_ERR_NO_FILE => 'Không có file nào được upload',
            UPLOAD_ERR_NO_TMP_DIR => 'Thiếu thư mục tạm',
            UPLOAD_ERR_CANT_WRITE => 'Không thể ghi file',
            UPLOAD_ERR_EXTENSION => 'File bị chặn bởi extension'
        ];
        
        return isset($messages[$errorCode]) ? $messages[$errorCode] : 'Lỗi upload không xác định';
    }
    
    /**
     * Schedule cleanup task
     */
    private function scheduleCleanup() {
        $lastCleanup = $this->getLastCleanupTime();
        $now = time();
        
        // Run cleanup if it's been more than CLEANUP_INTERVAL_MINUTES minutes
        if (($now - $lastCleanup) > (CLEANUP_INTERVAL_MINUTES * 60)) {
            $this->runCleanup();
        }
    }
    
    /**
     * Get last cleanup time
     */
    private function getLastCleanupTime() {
        $file = LOGS_DIR . 'last_cleanup.txt';
        if (file_exists($file)) {
            return (int)file_get_contents($file);
        }
        return 0;
    }
    
    /**
     * Run cleanup process
     */
    private function runCleanup() {
        $cleanupDirs = [
            UPLOAD_DIR,
            TEMP_DIR,
            DOWNLOADS_DIR
        ];
        
        $cutoffTime = time() - (CLEANUP_INTERVAL_MINUTES * 60);
        $deletedFiles = 0;
        $deletedSize = 0;
        
        foreach ($cleanupDirs as $dir) {
            $result = $this->cleanupDirectory($dir, $cutoffTime);
            $deletedFiles += $result['files'];
            $deletedSize += $result['size'];
        }
        
        // Log cleanup results
        $logMessage = sprintf(
            "[%s] Cleanup completed: %d files deleted, %.2f MB freed\n",
            date('Y-m-d H:i:s'),
            $deletedFiles,
            $deletedSize / (1024 * 1024)
        );
        
        file_put_contents(CLEANUP_LOG_FILE, $logMessage, FILE_APPEND | LOCK_EX);
        
        // Update last cleanup time
        file_put_contents(LOGS_DIR . 'last_cleanup.txt', time());
    }
    
    /**
     * Cleanup directory recursively
     */
    private function cleanupDirectory($dir, $cutoffTime) {
        $deletedFiles = 0;
        $deletedSize = 0;
        
        if (!is_dir($dir)) {
            return ['files' => 0, 'size' => 0];
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile() && $file->getMTime() < $cutoffTime) {
                $size = $file->getSize();
                if (unlink($file->getPathname())) {
                    $deletedFiles++;
                    $deletedSize += $size;
                }
            }
        }
        
        return ['files' => $deletedFiles, 'size' => $deletedSize];
    }
    
    /**
     * Get temporary file path
     */
    public function getTempPath($extension = '') {
        $filename = uniqid('temp_') . ($extension ? '.' . $extension : '');
        return TEMP_DIR . $filename;
    }
    
    /**
     * Clean up specific files
     */
    public function cleanup($files) {
        foreach ($files as $file) {
            if (file_exists($file)) {
                unlink($file);
            }
        }
    }
    
    /**
     * Get file size in human readable format
     */
    public function formatFileSize($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $factor = floor((strlen($bytes) - 1) / 3);
        return sprintf("%.2f %s", $bytes / pow(1024, $factor), $units[$factor]);
    }
}
?>
