// Enhanced iPa Signer JavaScript - Production Ready
class IPASignerApp {
    constructor() {
        this.maxFileSize = 500 * 1024 * 1024; // 500MB
        this.allowedTypes = {
            ipa: ['application/octet-stream', 'application/zip'],
            p12: ['application/x-pkcs12', 'application/octet-stream'],
            mobileprovision: ['application/octet-stream', 'text/xml']
        };
        this.uploadProgress = {};
        this.retryCount = 0;
        this.maxRetries = 3;
        this.init();
    }

    init() {
        this.setupFormHandlers();
        this.setupFileValidation();
        this.setupDragAndDrop();
        this.setupProgressTracking();
        this.loadSecurityToken();
        this.initThemeManager();
        this.setupConnectionCheck();
    }

    setupFormHandlers() {
        const form = document.getElementById('signingForm');
        if (form) {
            form.addEventListener('submit', (e) => this.handleFormSubmit(e));
        }

        // Enhanced password visibility toggle
        const toggleButtons = document.querySelectorAll('.toggle-password');
        toggleButtons.forEach(btn => {
            btn.addEventListener('click', (e) => this.togglePasswordVisibility(e));
        });

        // Auto-clear messages
        this.setupMessageAutoClear();
    }

    async handleFormSubmit(e) {
        e.preventDefault();
        
        try {
            // Validate form
            if (!this.validateForm()) {
                return;
            }

            // Check connection
            if (!await this.checkConnection()) {
                this.showMessage('Mất kết nối mạng. Vui lòng kiểm tra và thử lại.', 'error');
                return;
            }

            // Show enhanced loading with progress
            this.showLoadingWithProgress();
            
            // Create FormData with security token
            const formData = this.createFormData();
            
            // Send request with retry mechanism
            const result = await this.sendRequestWithRetry(formData);
            
            this.hideLoading();
            this.handleResponse(result);
            
        } catch (error) {
            this.hideLoading();
            this.handleError(error);
        }
    }

    createFormData() {
        const form = document.getElementById('signingForm');
        const formData = new FormData(form);
        
        // Add security token if available
        if (this.securityToken) {
            formData.append('security_token', this.securityToken);
        }
        
        // Add timestamp for debugging
        formData.append('submit_time', new Date().toISOString());
        
        return formData;
    }

    async sendRequestWithRetry(formData, attempt = 1) {
        try {
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 300000); // 5 minutes timeout
            
            const response = await fetch('process.php', {
                method: 'POST',
                body: formData,
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            return await response.json();
            
        } catch (error) {
            if (attempt < this.maxRetries && !error.name === 'AbortError') {
                this.updateProgressMessage(`Thử lại lần ${attempt + 1}...`);
                await this.delay(2000 * attempt); // Exponential backoff
                return this.sendRequestWithRetry(formData, attempt + 1);
            }
            throw error;
        }
    }

    validateForm() {
        const requiredFiles = ['ipaFile', 'p12File', 'mobileprovisionFile'];
        const password = document.getElementById('p12Password');
        
        // Check required files
        for (const fileId of requiredFiles) {
            const fileInput = document.getElementById(fileId);
            if (!fileInput || !fileInput.files.length) {
                this.showMessage(`Vui lòng chọn file ${fileId.replace('File', '')} hợp lệ`, 'error');
                return false;
            }
            
            // Enhanced file validation
            if (!this.validateFileDetailed(fileInput.files[0], fileId)) {
                return false;
            }
        }
        
        // Check password
        if (!password || !password.value.trim()) {
            this.showMessage('Vui lòng nhập mật khẩu cho chứng chỉ P12', 'error');
            password.focus();
            return false;
        }
        
        return true;
    }

    validateFileDetailed(file, type) {
        // Size validation
        if (file.size > this.maxFileSize) {
            this.showMessage(`File ${file.name} quá lớn. Tối đa ${this.formatFileSize(this.maxFileSize)}`, 'error');
            return false;
        }
        
        // Type validation based on extension and MIME type
        const extension = '.' + file.name.split('.').pop().toLowerCase();
        const expectedExtensions = {
            'ipaFile': ['.ipa'],
            'p12File': ['.p12'],
            'mobileprovisionFile': ['.mobileprovision']
        };
        
        if (expectedExtensions[type] && !expectedExtensions[type].includes(extension)) {
            this.showMessage(`File ${file.name} không đúng định dạng ${expectedExtensions[type].join(', ')}`, 'error');
            return false;
        }
        
        return true;
    }

    setupFileValidation() {
        const fileInputs = document.querySelectorAll('input[type="file"]');
        fileInputs.forEach(input => {
            input.addEventListener('change', (e) => {
                if (e.target.files.length > 0) {
                    this.validateFileDetailed(e.target.files[0], e.target.id);
                    this.updateFileDisplay(e.target);
                }
            });
        });
    }

    updateFileDisplay(input) {
        const file = input.files[0];
        if (!file) return;
        
        const displayElement = input.parentElement.querySelector('.file-display');
        if (displayElement) {
            displayElement.innerHTML = `
                <div class="file-info">
                    <i class="fas fa-file"></i>
                    <span class="file-name">${file.name}</span>
                    <span class="file-size">${this.formatFileSize(file.size)}</span>
                </div>
            `;
        }
    }

    setupDragAndDrop() {
        const dropZones = document.querySelectorAll('.file-upload-wrapper');
        dropZones.forEach(zone => {
            zone.addEventListener('dragover', (e) => this.handleDragOver(e));
            zone.addEventListener('dragleave', (e) => this.handleDragLeave(e));
            zone.addEventListener('drop', (e) => this.handleDrop(e));
        });
    }

    handleDragOver(e) {
        e.preventDefault();
        e.currentTarget.classList.add('drag-over');
    }

    handleDragLeave(e) {
        e.currentTarget.classList.remove('drag-over');
    }

    handleDrop(e) {
        e.preventDefault();
        e.currentTarget.classList.remove('drag-over');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            const fileInput = e.currentTarget.querySelector('input[type="file"]');
            if (fileInput) {
                fileInput.files = files;
                this.validateFileDetailed(files[0], fileInput.id);
                this.updateFileDisplay(fileInput);
            }
        }
    }

    showLoadingWithProgress() {
        const existingModal = document.querySelector('.loading-modal');
        if (existingModal) {
            existingModal.remove();
        }
        
        const modal = document.createElement('div');
        modal.className = 'loading-modal';
        modal.innerHTML = `
            <div class="loading-content">
                <div class="loading-spinner"></div>
                <h3>Đang xử lý...</h3>
                <div class="progress-bar">
                    <div class="progress-fill" style="width: 0%"></div>
                </div>
                <p class="progress-message">Đang tải file lên...</p>
                <div class="loading-steps">
                    <div class="step active">📤 Upload files</div>
                    <div class="step">🔒 Xác thực chứng chỉ</div>
                    <div class="step">✍️ Ký IPA</div>
                    <div class="step">✅ Hoàn thành</div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Simulate progress
        this.simulateProgress();
    }

    simulateProgress() {
        const progressFill = document.querySelector('.progress-fill');
        const steps = document.querySelectorAll('.loading-steps .step');
        
        let progress = 0;
        const interval = setInterval(() => {
            progress += Math.random() * 15;
            if (progress > 90) progress = 90; // Don't complete until real response
            
            if (progressFill) {
                progressFill.style.width = progress + '%';
            }
            
            // Update active step
            const currentStep = Math.floor(progress / 25);
            steps.forEach((step, index) => {
                step.classList.toggle('active', index <= currentStep);
                step.classList.toggle('completed', index < currentStep);
            });
            
            if (progress >= 90) {
                clearInterval(interval);
            }
        }, 500);
        
        this.progressInterval = interval;
    }

    updateProgressMessage(message) {
        const messageElement = document.querySelector('.progress-message');
        if (messageElement) {
            messageElement.textContent = message;
        }
    }

    hideLoading() {
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
        }
        
        const modal = document.querySelector('.loading-modal');
        if (modal) {
            // Complete the progress
            const progressFill = modal.querySelector('.progress-fill');
            const steps = modal.querySelectorAll('.loading-steps .step');
            
            if (progressFill) progressFill.style.width = '100%';
            steps.forEach(step => step.classList.add('completed'));
            
            setTimeout(() => {
                modal.remove();
            }, 1000);
        }
    }

    handleResponse(data) {
        if (data.success) {
            this.showSuccessResponse(data);
        } else {
            this.showMessage(data.message || 'Có lỗi xảy ra', 'error');
        }
    }

    showSuccessResponse(data) {
        const modal = document.createElement('div');
        modal.className = 'success-modal';
        modal.innerHTML = `
            <div class="success-content">
                <div class="success-header">
                    <i class="fas fa-check-circle"></i>
                    <h2>Ký IPA thành công!</h2>
                </div>
                <div class="file-info">
                    <p><strong>File:</strong> ${data.signed_file}</p>
                    ${data.file_size ? `<p><strong>Kích thước:</strong> ${data.file_size}</p>` : ''}
                    ${data.signed_at ? `<p><strong>Thời gian:</strong> ${data.signed_at}</p>` : ''}
                </div>
                <div class="download-actions">
                    <a href="${data.download_url}" class="btn btn-primary" download>
                        <i class="fas fa-download"></i> Tải về
                    </a>
                    ${data.install_url ? `
                        <a href="${data.install_url}" class="btn btn-secondary">
                            <i class="fas fa-mobile-alt"></i> Cài đặt trên iOS
                        </a>
                    ` : ''}
                </div>
                <button class="btn btn-outline close-modal">Đóng</button>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Bind close event
        modal.querySelector('.close-modal').addEventListener('click', () => {
            modal.remove();
        });
        
        // Auto close after 30 seconds
        setTimeout(() => {
            if (document.body.contains(modal)) {
                modal.remove();
            }
        }, 30000);
    }

    handleError(error) {
        console.error('Signing error:', error);
        
        let message = 'Có lỗi xảy ra khi xử lý yêu cầu';
        
        if (error.name === 'AbortError') {
            message = 'Yêu cầu bị hủy do timeout. Vui lòng thử lại.';
        } else if (error.message.includes('Failed to fetch')) {
            message = 'Không thể kết nối đến server. Vui lòng kiểm tra mạng.';
        } else if (error.message) {
            message = error.message;
        }
        
        this.showMessage(message, 'error');
    }

    showMessage(message, type = 'info') {
        // Remove existing messages
        const existingMessages = document.querySelectorAll('.message-toast');
        existingMessages.forEach(msg => msg.remove());
        
        const toast = document.createElement('div');
        toast.className = `message-toast ${type}`;
        toast.innerHTML = `
            <div class="message-content">
                <i class="fas ${this.getMessageIcon(type)}"></i>
                <span>${message}</span>
                <button class="close-message">&times;</button>
            </div>
        `;
        
        document.body.appendChild(toast);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (document.body.contains(toast)) {
                toast.remove();
            }
        }, 5000);
        
        // Manual close
        toast.querySelector('.close-message').addEventListener('click', () => {
            toast.remove();
        });
    }

    getMessageIcon(type) {
        const icons = {
            'success': 'fa-check-circle',
            'error': 'fa-exclamation-circle',
            'warning': 'fa-exclamation-triangle',
            'info': 'fa-info-circle'
        };
        return icons[type] || icons.info;
    }

    setupMessageAutoClear() {
        // Clear messages when user starts interacting with form
        const inputs = document.querySelectorAll('input, textarea, select');
        inputs.forEach(input => {
            input.addEventListener('focus', () => {
                const messages = document.querySelectorAll('.message-toast.error');
                messages.forEach(msg => msg.remove());
            });
        });
    }

    togglePasswordVisibility(e) {
        const button = e.currentTarget;
        const input = button.parentElement.querySelector('input[type="password"], input[type="text"]');
        
        if (input) {
            const isPassword = input.type === 'password';
            input.type = isPassword ? 'text' : 'password';
            button.innerHTML = isPassword ? 
                '<i class="fas fa-eye-slash"></i>' : 
                '<i class="fas fa-eye"></i>';
        }
    }

    async loadSecurityToken() {
        try {
            const response = await fetch('get_token.php');
            const data = await response.json();
            if (data.token) {
                this.securityToken = data.token;
            }
        } catch (error) {
            console.log('Security token not available:', error);
        }
    }

    initThemeManager() {
        const savedTheme = localStorage.getItem('theme') || 'light';
        this.setTheme(savedTheme);
        
        const themeToggle = document.getElementById('themeToggle');
        if (themeToggle) {
            themeToggle.addEventListener('click', () => this.toggleTheme());
        }
    }

    setTheme(theme) {
        document.body.classList.toggle('dark-mode', theme === 'dark');
        const themeIcon = document.getElementById('themeIcon');
        if (themeIcon) {
            themeIcon.className = theme === 'dark' ? 'fas fa-sun' : 'fas fa-moon';
        }
        localStorage.setItem('theme', theme);
    }

    toggleTheme() {
        const isDark = document.body.classList.contains('dark-mode');
        this.setTheme(isDark ? 'light' : 'dark');
    }

    async checkConnection() {
        try {
            const response = await fetch('status.php', {
                method: 'HEAD',
                cache: 'no-cache'
            });
            return response.ok;
        } catch (error) {
            return false;
        }
    }

    setupConnectionCheck() {
        // Check connection every 30 seconds
        setInterval(async () => {
            const isOnline = await this.checkConnection();
            const indicator = document.querySelector('.connection-indicator');
            
            if (indicator) {
                indicator.classList.toggle('offline', !isOnline);
                indicator.title = isOnline ? 'Đã kết nối' : 'Mất kết nối';
            }
        }, 30000);
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 B';
        const k = 1024;
        const sizes = ['B', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// Initialize app when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.ipaSignerApp = new IPASignerApp();
});

// Service Worker registration for offline capability
if ('serviceWorker' in navigator) {
    window.addEventListener('load', () => {
        navigator.serviceWorker.register('/sw.js')
            .then(registration => {
                console.log('SW registered: ', registration);
            })
            .catch(registrationError => {
                console.log('SW registration failed: ', registrationError);
            });
    });
}
