// Theme Management
class ThemeManager {
    constructor() {
        this.init();
    }

    init() {
        const savedTheme = localStorage.getItem('theme') || 'light';
        this.setTheme(savedTheme);
        this.bindEvents();
    }

    setTheme(theme) {
        document.body.classList.toggle('dark-mode', theme === 'dark');
        const themeIcon = document.getElementById('themeIcon');
        if (themeIcon) {
            themeIcon.className = theme === 'dark' ? 'fas fa-sun' : 'fas fa-moon';
        }
        localStorage.setItem('theme', theme);
    }

    toggleTheme() {
        const isDark = document.body.classList.contains('dark-mode');
        this.setTheme(isDark ? 'light' : 'dark');
    }

    bindEvents() {
        const themeToggle = document.getElementById('themeToggle');
        if (themeToggle) {
            themeToggle.addEventListener('click', () => this.toggleTheme());
        }
    }
}

// File Upload Handler
class FileUploadHandler {
    constructor() {
        this.maxSize = 100 * 1024 * 1024; // 100MB
        this.init();
    }

    init() {
        this.bindEvents();
    }

    bindEvents() {
        // IPA File Upload
        const ipaUpload = document.getElementById('ipaUpload');
        const ipaFile = document.getElementById('ipaFile');
        this.setupFileUpload(ipaUpload, ipaFile, ['.ipa']);

        // P12 File Upload
        const p12Upload = document.getElementById('p12Upload');
        const p12File = document.getElementById('p12File');
        this.setupFileUpload(p12Upload, p12File, ['.p12']);

        // Mobileprovision File Upload
        const mpUpload = document.getElementById('mobileprovisionUpload');
        const mpFile = document.getElementById('mobileprovisionFile');
        this.setupFileUpload(mpUpload, mpFile, ['.mobileprovision']);

        // Password Toggle
        const togglePassword = document.getElementById('togglePassword');
        const passwordInput = document.getElementById('p12Password');
        if (togglePassword && passwordInput) {
            togglePassword.addEventListener('click', () => {
                const type = passwordInput.type === 'password' ? 'text' : 'password';
                passwordInput.type = type;
                togglePassword.innerHTML = type === 'password' ? 
                    '<i class="fas fa-eye"></i>' : '<i class="fas fa-eye-slash"></i>';
            });
        }
    }

    setupFileUpload(uploadArea, fileInput, allowedExtensions) {
        if (!uploadArea || !fileInput) return;

        uploadArea.addEventListener('click', () => fileInput.click());

        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });

        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });

        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                this.handleFileSelect(files[0], uploadArea, fileInput, allowedExtensions);
            }
        });

        fileInput.addEventListener('change', (e) => {
            if (e.target.files.length > 0) {
                this.handleFileSelect(e.target.files[0], uploadArea, fileInput, allowedExtensions);
            }
        });
    }

    handleFileSelect(file, uploadArea, fileInput, allowedExtensions) {
        const fileName = file.name.toLowerCase();
        const isValidExtension = allowedExtensions.some(ext => fileName.endsWith(ext));

        if (!isValidExtension) {
            this.showError(`Chỉ chấp nhận file ${allowedExtensions.join(', ')}`);
            return;
        }

        if (file.size > this.maxSize) {
            this.showError('File quá lớn. Kích thước tối đa là 100MB.');
            return;
        }

        // Update UI
        uploadArea.classList.add('uploaded');
        const fileInfo = uploadArea.querySelector('.file-info');
        const fileNameSpan = uploadArea.querySelector('.file-name');
        const fileSizeSpan = uploadArea.querySelector('.file-size');

        if (fileInfo && fileNameSpan && fileSizeSpan) {
            fileNameSpan.textContent = file.name;
            fileSizeSpan.textContent = this.formatFileSize(file.size);
            fileInfo.classList.remove('d-none');
        }

        // Set the file to input
        const dt = new DataTransfer();
        dt.items.add(file);
        fileInput.files = dt.files;

        // Hide upload text
        const uploadText = uploadArea.querySelector('h5, p');
        if (uploadText) {
            uploadArea.querySelector('h5').style.display = 'none';
            uploadArea.querySelector('p').style.display = 'none';
        }
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    showError(message) {
        // Create toast notification
        const toast = document.createElement('div');
        toast.className = 'toast align-items-center text-white bg-danger border-0 position-fixed';
        toast.style.cssText = 'top: 20px; right: 20px; z-index: 9999;';
        toast.innerHTML = `
            <div class="d-flex">
                <div class="toast-body">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        `;
        document.body.appendChild(toast);
        
        const bsToast = new bootstrap.Toast(toast);
        bsToast.show();
        
        setTimeout(() => {
            if (toast.parentNode) {
                toast.parentNode.removeChild(toast);
            }
        }, 5000);
    }
}

// Form Handler
class FormHandler {
    constructor() {
        this.form = document.getElementById('signForm');
        this.progressContainer = document.getElementById('progressContainer');
        this.progressBar = document.getElementById('progressBar');
        this.progressText = document.getElementById('progressText');
        this.progressPercent = document.getElementById('progressPercent');
        this.signBtn = document.getElementById('signBtn');
        
        this.init();
    }

    init() {
        if (this.form) {
            this.form.addEventListener('submit', (e) => this.handleSubmit(e));
        }
    }

    async handleSubmit(e) {
        e.preventDefault();

        if (!this.validateForm()) {
            return;
        }

        this.showProgress();
        
        try {
            const formData = new FormData(this.form);
            
            // Simulate progress
            this.updateProgress(10, 'Đang upload files...');
            
            const response = await fetch('process.php', {
                method: 'POST',
                body: formData
            });

            this.updateProgress(50, 'Đang xử lý...');

            const result = await response.json();
            
            if (result.success) {
                this.updateProgress(100, 'Hoàn thành!');
                setTimeout(() => {
                    window.location.href = `success.php?file=${encodeURIComponent(result.signed_file)}`;
                }, 1000);
            } else {
                throw new Error(result.message || 'Có lỗi xảy ra trong quá trình xử lý');
            }
        } catch (error) {
            console.error('Error:', error);
            this.hideProgress();
            this.showError(error.message || 'Có lỗi xảy ra, vui lòng thử lại');
        }
    }

    validateForm() {
        const ipaFile = document.getElementById('ipaFile').files[0];
        const p12File = document.getElementById('p12File').files[0];
        const mpFile = document.getElementById('mobileprovisionFile').files[0];
        const password = document.getElementById('p12Password').value;

        if (!ipaFile) {
            this.showError('Vui lòng chọn file IPA');
            return false;
        }

        if (!p12File) {
            this.showError('Vui lòng chọn file chứng chỉ P12');
            return false;
        }

        if (!mpFile) {
            this.showError('Vui lòng chọn file Provisioning Profile');
            return false;
        }

        if (!password.trim()) {
            this.showError('Vui lòng nhập mật khẩu cho chứng chỉ P12');
            return false;
        }

        return true;
    }

    showProgress() {
        this.form.style.display = 'none';
        this.progressContainer.classList.remove('d-none');
        this.signBtn.disabled = true;
    }

    hideProgress() {
        this.form.style.display = 'block';
        this.progressContainer.classList.add('d-none');
        this.signBtn.disabled = false;
    }

    updateProgress(percent, text) {
        if (this.progressBar) {
            this.progressBar.style.width = percent + '%';
        }
        if (this.progressText) {
            this.progressText.textContent = text;
        }
        if (this.progressPercent) {
            this.progressPercent.textContent = percent + '%';
        }
    }

    showError(message) {
        const toast = document.createElement('div');
        toast.className = 'toast align-items-center text-white bg-danger border-0 position-fixed';
        toast.style.cssText = 'top: 20px; right: 20px; z-index: 9999;';
        toast.innerHTML = `
            <div class="d-flex">
                <div class="toast-body">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        `;
        document.body.appendChild(toast);
        
        const bsToast = new bootstrap.Toast(toast);
        bsToast.show();
        
        setTimeout(() => {
            if (toast.parentNode) {
                toast.parentNode.removeChild(toast);
            }
        }, 5000);
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new ThemeManager();
    new FileUploadHandler();
    new FormHandler();
    
    // Add animations
    const mainCard = document.querySelector('.main-card');
    if (mainCard) {
        mainCard.classList.add('fade-in');
    }
    
    const featureCards = document.querySelectorAll('.feature-card');
    featureCards.forEach((card, index) => {
        setTimeout(() => {
            card.classList.add('slide-up');
        }, 200 * (index + 1));
    });
});
